package fccsc.manager;

import java.io.InputStream;
import java.io.OutputStream;
import java.net.Socket;

import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;

import fccsc.manager.crypto.CryptoData;
import fccsc.manager.crypto.MessageCryptography;


public final class MessageThread
	implements Runnable
{
	/** OK: message will be processed. */
	public static final String CLOSE000 = "CLOSE   ";

	/** Warning: message will be truncated and processed. */
	public static final String CLOSE001 = "CLOSE001";

	/** Error: message will not be processed. */
	public static final String CLOSE002 = "CLOSE002";

	/** Error: message length field is not numeric. */
	public static final String CLOSE003 = "CLOSE003";

	/** Error: Control Block contains garbage !?@#. */
	public static final String CLOSE004 = "CLOSE004";


    private static Logger logger = LogManager.getLogger( MessageThread.class );  // PBSC palermor IR84565

	private MessageHandler messageHandler = null;
	private Socket         socket         = null;
	private InputStream    stream_in      = null;
	private OutputStream   stream_out     = null;
	private StringBuffer   bufferIn       = new StringBuffer();
	private byte []        bytesIn        = new byte[6];
	private String         closeCode      = CLOSE000;



	public
	MessageThread( Socket p_socket )
	{
		this.socket = p_socket;
	}


	public void
	run()
	{
		if ( logger.isDebugEnabled() )
		{
			logger.debug( "Processing START ..." );
		}

		try
		{
			if ( socket != null )
			{
				// create our reading from and writing to streams
				stream_in  = this.socket.getInputStream();
				stream_out = this.socket.getOutputStream();

				socketRead();
				socketClose();

				if ( (this.closeCode != null) &&
					 (this.closeCode.equalsIgnoreCase( CLOSE000 ) ||
					  this.closeCode.equalsIgnoreCase( CLOSE001 )  ) )
				{
		    		boolean    useEncryption = false;
					CryptoData crypdata      = null;
					String     request       = this.bufferIn.toString();

					if ( logger.isDebugEnabled() )
					{
						logger.debug( "Received data [{} / {}]", this.bufferIn.length(), this.bufferIn.toString() );  // PBSC palermor IR84565
					}

					// decryption process
					//
					if ( this.bufferIn.charAt( 15 ) == '3' )
					{
						crypdata      = MessageCryptography.decrypt( this.bytesIn );
						request       = new String( crypdata.getData() );
						useEncryption = true;
					}

					this.messageHandler = new MessageHandler();
					this.messageHandler.processRequest( request );
//String   response     = request;
//String   callbackIp   = "localhost"; //"10.0.0.4";
//int      callbackPort = 5200;
//System.out.println( Thread.currentThread().getName() + " [" + new java.text.SimpleDateFormat( "hh:MM:ss:SSS]" ).format( new Date() ) + "]" );

					// now get the actual message and the actual response data
					// and its callback information to send back to ...
					//
					IMessage message      = this.messageHandler.getMessage();
					String   response     = message.getResponse();
					String   callbackIp   = message.getCallBackIpAddress();
					int      callbackPort = message.getCallBackIpPort();
					byte [] responseBytes = response.getBytes();

					// encryption process
					//
					if ( (useEncryption) && (response.length() > 0) )
					{
						responseBytes = MessageCryptography.encryptWithSessionKey(
															response.getBytes(),
														    crypdata.getSessionKey() );
					}

					if ( logger.isDebugEnabled() )
					{
						logger.debug( "Sending response [{} / {}]", response.length(), response );  // PBSC palermor IR84565
					}

					// send data back to callback number
					//
					MessageSender sender = new MessageSender( callbackIp, callbackPort );
					sender.send( responseBytes );
				}
			}
			else
			{
				if ( logger.isDebugEnabled() )
				{
					logger.debug( "Socket is NULL !" );
				}
			}
		}
		catch ( Exception e)
		{
			e.printStackTrace();

			logger.error( e.getMessage() );
		}
		finally
		{
			try
			{
				if ( this.stream_in  != null ) { this.stream_in.close();  }
				if ( this.stream_out != null ) { this.stream_out.close(); }
				if ( this.socket     != null ) { this.socket.close();     }
			}
			catch (Exception e2)
			{
				e2.printStackTrace();

				logger.error( e2.getMessage() );
			}
		}

		if ( logger.isDebugEnabled() )
		{
			logger.debug( "Processing COMPLETE." );
		}
	}


	private void
	socketRead()
	{
		int    iCount = 0;
		int    iSize  = 0;
		int    c;
		String sSize  = "";

		try
		{
			if ( logger.isDebugEnabled() )
			{
				logger.debug( "Receiving ..." );
			}

			while ( (c = stream_in.read() ) != -1 )
			{
				iCount++;
				//System.out.println( "c [" + c + " = " + (char)c + "] " + iCount );

				// store the byte
				this.bytesIn[ iCount - 1 ] = (byte) c;

				// store the current byte as a char
				this.bufferIn.append( (char) c );

				// read and store message size from first 6 bytes
				//
				if ( bufferIn.length() == 6 )
				{
					//System.out.println( "Storing buffer size" );
					sSize = bufferIn.substring( 0, 6 );
					iSize = Integer.parseInt( sSize );

					// now that we know how big the message we are to receive
					// reset the size of the array and copy the contents of the
					// first 6 bytes we already have read
					//
					byte [] arrTemp = this.bytesIn;

					this.bytesIn = new byte[ iSize ];

					for (int i = 0; i < 6; i++)
					{
						this.bytesIn[i] = arrTemp[i];
					}
				}

				// check if the chars received has been met by our counter
				//
				if ( iSize == iCount )
				{
					if ( iSize == 22 )  // CRYP handshake
					{
						if ( logger.isDebugEnabled() )
						{
							logger.debug( "Sending CRYP handshake [000017TCP CRYP000]" );
						}

						iCount = 0;
						this.bytesIn  = new byte[ 6 ];
						this.bufferIn = new StringBuffer();
						stream_out.write( new String( "000017TCP CRYP000" ).getBytes() );
					}
					else // DATA message
					{
						break;
					}
				}
			}
			//System.out.println( "<END>" );
			//System.out.println( "length:" + bufferIn.length() );
			//System.out.println( "buffer[" + bufferIn.toString() + "]" );
		}
		catch ( Exception e)
		{
			//
			// ERROR - timeout or connection error or ...
			//
			closeCode = CLOSE002;
			e.printStackTrace();

			logger.error( "Received [{}] Error: message will not be processed. (Timeout or connection error or ...)", CLOSE002 );  // PBSC palermor IR84565
			logger.error( e.getMessage() );
		}

		// if length is zero, then no exceptions occurred
		if ( closeCode.trim().length() == 0 )
		{
			int lengthStated   = 0;
			int lengthRecieved = this.bufferIn.length();

			try
			{
				lengthStated = Integer.parseInt( this.bufferIn.substring( 0, 6 ) );
			}
			catch (NumberFormatException e2)
			{
				//
				// ERROR - the message length at the start of the message is not numeric,
				// or it has some other error.
				//
				closeCode = CLOSE003;
				e2.printStackTrace();

				logger.error( "Received [{}] Error: message length field is not numeric.", CLOSE003 );  // PBSC palermor IR84565
				logger.error( e2.getMessage() );
			}

		    //System.out.println( DATA + "Received/Stated: " + lengthRecieved + "/" + lengthStated );
			if ( logger.isDebugEnabled() )
			{
				logger.debug( "Received / Stated: [{} / {}] bytes", lengthRecieved, lengthStated );  // PBSC palermor IR84565
			}

			// if length is zero, stated message length is numeric
			if ( closeCode.trim().length() == 0 )
			{
				//
				// BLANK - no error.
				if ( lengthRecieved == lengthStated )
				{
					closeCode = CLOSE000;

					if ( logger.isDebugEnabled() )
					{
						logger.debug( "Received [{}] OK: message will be processed.", CLOSE000 );  // PBSC palermor IR84565
					}
				}
				//
				// WARNING - more data received than was indicated by the message
				// length field at the start of the message. Message will be truncated
				// to the stated length, and processed.
				else if ( lengthRecieved  > lengthStated )
				{
					closeCode = CLOSE001;
					this.bufferIn = new StringBuffer( this.bufferIn.substring( 0, lengthStated ) );

					logger.error( "Received [{}] Warning: message will be truncated and processed.", CLOSE001 );  // PBSC palermor IR84565
					logger.error( "Received [{}] Received/Stated [{}/{}]", CLOSE001, lengthRecieved, lengthStated );  // PBSC palermor IR84565
				}
				//
				// ERROR - there was less data then was indicated by the message
				// length field at the start of the message, OR there was a timeout
				// before all of the message was received. Message will NOT be processed.
				else if ( lengthRecieved  < lengthStated )
				{
				    closeCode = CLOSE002;

					logger.error( "Received [{}] Error: message will not be processed.", CLOSE002 );  // PBSC palermor IR84565
					logger.error( "Received [{}] Received/Stated [{}/{}]", CLOSE002, lengthRecieved, lengthStated );  // PBSC palermor IR84565
				}
				//
				// ERROR - the Control Block contains garbage.
				else
				{
					closeCode = CLOSE004;

					logger.error( "Received [{}] Error: Control Block contains garbage !?@#.", CLOSE004 );  // PBSC palermor IR84565
				}
			}
		}
	}


	private void
	socketClose()
	{
		try
		{
			if ( logger.isDebugEnabled() )
			{
				logger.debug( "Sending  [{}].", this.closeCode );  // PBSC palermor IR84565
			}

			stream_out.write( this.closeCode.getBytes() );
		}
		catch ( Exception e)
		{
			e.printStackTrace();
		    logger.error( e.getMessage() );
		}
		finally
		{
			try
			{
				stream_out.flush();
				stream_out.close();
			}
			catch (Exception e2)
			{
				e2.printStackTrace();
			}
		}
	}
}